/*******************************************************************
*
*  DESCRIPTION: class Driver : Implements Driver class initialization and finalization functions + model's
*  own input/output port control functions.
*
*  AUTHOR:    Mohammad Moallemi
*
*  EMAIL: mailto: moallemi@sce.carleton.ca
*
*  DATE: 5/3/2009
*  Date Mod : 11/08/2009
*
*******************************************************************/

// RobotVehicle Model (Dieynaba) include files
#include <driver.h>

#include "In2.h"
#include "Out2.h"
#include "Out3.h"
#include <nxt4/NXT++.h>	//NXT++ driver
#include <native/mutex.h>
#include "commonH.h"

RT_MUTEX mutex_nxt;

/*******************************************************************
* Function Name: Driver::InitHardware
* Description: Opens hardware connections  
*******************************************************************/
const bool Driver::initHardware()
{
    // Open the connection to the LEGO robot
    int open = NXT::Open();
    // Return if we cannot open the connection
    if (open == 0)
    {
       std::cout << "Could not connect" << std::endl;
       return false;
    }
    else 
    {
       std::cout << "USB Opened" << std::endl;
    }

    // Set Port 1 for Light Sensor 1
    NXT::Sensor::SetLight(IN_1, false); //tell the NXT that the Light Sensor is in port 1 
    std::cout << "Light Sensor 1 initialized" << std::endl;
	
	if(rt_mutex_create(&mutex_nxt,"MyMutex") < 0){
	    std::cout << "Could not open NXT Mutex" << std::endl;
	    return false;
	}
	return true;
}

/*******************************************************************
* Function Name: Driver::CloseHardware
* Description: Closes hardware connections *******************************************************************/
const bool Driver::closeHardware()
{
	//Stop motor on port B
	NXT::Motor::Stop(OUT_B,true);
     
    //**** CHANGE MADE *** (Addition)
    //Stop motor on port C
	NXT::Motor::Stop(OUT_C,true);

    // Close the connection to the LEGO robot
    NXT::Close(); //close the NXT
	
	rt_mutex_delete(&mutex_nxt);
	
	return true;
}


/*******************************************************************
* Function Name: In2::pDriver
* Description: Gets Light Sensor 1 Value
*******************************************************************/
bool In2::pDriver(Value &value)
{
  //Get Light Sensor 1 Value and save it in variable "value" 
	rt_mutex_acquire(&mutex_nxt,TM_INFINITE);
	    double percent = NXT::Sensor::GetValue(IN_1);
 	rt_mutex_release(&mutex_nxt);
    
    //if we get equal or less than 50% light then there's a detected line pattern
    if(percent <= 25)
    	value = ALL_DARK;
    else
	    value = percent<=50 ? DARK : BRIGHT;

	return percent<=50 ? true : false;
}



/*******************************************************************
* Function Name: Out2::pDriver
* Description: sends outputs to right motor
*******************************************************************/
bool Out2::pDriver(Value &value)
{
	rt_mutex_acquire(&mutex_nxt,TM_INFINITE);
		switch((int)value){
		    case COMMON_H::O_STOP: // Stop Right Motor on port B 
				NXT::Motor::Stop(OUT_B,true);
				break;
		
			case COMMON_H::O_GO_FWD: // Move Right Motor on port B Forward or ClockWise at Power Level 50
				NXT::Motor::Stop(OUT_B,true);
			    NXT::Motor::SetForward(OUT_B, 30); 
				break;
		
		  	case COMMON_H::O_GO_REV: // Move Right Motor on port B Backward or Counter-ClockWise at Power Level 50
				NXT::Motor::Stop(OUT_B,true);
				NXT::Motor::SetReverse(OUT_B, 30);
				break;
		 };
	rt_mutex_release(&mutex_nxt);
 	
 	return true;
}


/*******************************************************************
* Function Name: Out3::pDriver
* Description: sends outputs to left motor
*******************************************************************/
bool Out3::pDriver(Value &value)
{
	rt_mutex_acquire(&mutex_nxt,TM_INFINITE);
		 switch((int)value){
		    case COMMON_H::O_STOP: // Stop Left Motor on port C 
				NXT::Motor::Stop(OUT_C,true);
				break;
		
			case COMMON_H::O_GO_FWD: // Move Left Motor on port C Forward or ClockWise at Power Level 50
				NXT::Motor::Stop(OUT_C,true);
			    NXT::Motor::SetForward(OUT_C, 30); 
				break;
		
		  	case COMMON_H::O_GO_REV: // Move Right Motor on port C Backward or Counter-ClockWise at Power Level 50
				NXT::Motor::Stop(OUT_C,true);
				NXT::Motor::SetReverse(OUT_C, 30);
				break;
		 };
	rt_mutex_release(&mutex_nxt);
	
	return true;
}
